﻿using System;
using CoreLib;
using CoreLib.Commands;
using CoreLib.Util.Extensions;
using CrowdControl.Delegates.Effects;
using PugMod;
using UnityEngine;
using Object = UnityEngine.Object;

namespace CrowdControl
{
    /// <summary>
    /// The main Crowd Control mod class.
    /// </summary>
    public class CrowdControl : IMod 
    {
        // Mod Details
        public const string MOD_GUID = "WarpWorld.CrowdControl";
        public const string MOD_NAME = "Crowd Control";
        public const string MOD_VERSION = "1.0.1";
    
        public static float DeltaTime => Time.deltaTime / Time.timeScale;

        /// <summary>The singleton instance of the game mod.</summary>
        internal static CrowdControl Instance { get; private set; } = null!;

        /// <summary>The game state manager object.</summary>
        public GameStateManager GameStateManager { get; private set; } = null!;
    
        /// <summary>The effect class loader.</summary>
        public EffectLoader EffectLoader { get; private set; } = null!;
        
        /// <summary>The Harmony instance for patching.</summary>
        private readonly HarmonyLib.Harmony m_harmony = new(MOD_GUID);

        /// <summary>
        /// Gets a value indicating whether the client is connected.
        /// </summary>
        public bool ClientConnected => Client.Connected;

        public NetworkClient Client { get; private set; }
    
        public Scheduler Scheduler { get; private set; }
    
        private const float GAME_STATUS_UPDATE_INTERVAL = 1f;
        private float m_gameStatusUpdateTimer;

        public void EarlyInit()
        {
            CoreLibMod.LoadModules(typeof(CommandsModule));
            LoadedMod modInfo = this.GetModInfo();
            CommandsModule.AddCommands(modInfo.ModId, MOD_NAME);
        }

        public void Init()
        {
            Instance = this;

            Debug.Log($"Loaded {MOD_GUID}. Patching.");
            m_harmony.PatchAll();

            Debug.Log("Initializing Crowd Control");

            try
            {
                GameStateManager = new(this);
                Client = new(this);
                EffectLoader = new(this, Client);
                Scheduler = new(this, Client);
            }
            catch (Exception e)
            {
                Debug.LogError($"Crowd Control Init Error: {e}");
            }

            Debug.Log("Crowd Control Initialized");
        }

        public void Shutdown() { }

        public void ModObjectLoaded(Object obj) { }

        public void Update()
        {
            m_gameStatusUpdateTimer += DeltaTime;
            if (m_gameStatusUpdateTimer >= GAME_STATUS_UPDATE_INTERVAL)
            {
                GameStateManager.UpdateGameState();
                m_gameStatusUpdateTimer = 0f;
            }
        
            Scheduler?.Tick();
        }
    }
}
